// UTF-8 encoding
// Copyright 2009 Florent Lioult.

/**
 * @file	Jeu.cpp
 * @author	Florent Lioult
 * @date	2009
 */

#include <fstream>

#include "macro.hpp"
#include "Jeu.hpp"

using namespace std;

namespace AneRouge {

	Jeu::Jeu( const DescripteurJeu & descripteur ) :
		descripteur_( descripteur ),
		pAneRouge_( 0 )
	{}
	
	Jeu::~Jeu()
	{
		nettoyer();
	}
	
	istream & Jeu::deserialiser( std::istream & entree )
	{
		nettoyer();
		try
		{
			vector< string > lignes;
			
			// On lit complètement flux.

			size_t largeur = std::numeric_limits< size_t >::max();
			size_t hauteur = 0;

			while ( not entree.eof() )
			{
				string ligne;
				getline( entree, ligne );
				if ( not entree.eof() )
				{
					if (
						largeur != std::numeric_limits< size_t >::max() &&
						largeur != ligne.length() )
					{
						throw runtime_error( "Flux de données malformé : largeur non constante" );
					}
					lignes.push_back ( ligne );
					largeur = ligne.length();
					++hauteur;
				}
			}
			
			// Le programme peut traiter un plateau vide mais ce n'est pas très utile.

			if ( largeur == 0 or hauteur == 0 )
			{
				throw runtime_error( "Flux de données malformé : plateau vide" );
			}

			// On se ménage de la place pour une bordure autour des pièces.
			
			plateau_.redimensionner( largeur + 2, hauteur + 2 );

			// On remplit les cases intérieurs du plateau avec le contenu du flux.

			Piece * pBordure = new Piece( descripteur_.nomBordure, plateau_ );
			pieces_.insert( AnnuairePieces::value_type( pBordure->getNom(),  pBordure ) );

			for ( size_t y = 0; y < hauteur + 2; ++y )
			{
				for ( size_t x = 0; x < largeur + 2; ++x )
				{
					Position position( x, y );

					if ( x == 0 or x == largeur + 1 or y == 0 or y == hauteur + 1 )
					{
						pBordure->completerGeometrie( position );
					}
					else
					{
						char nomPiece = lignes[ y - 1 ][ x - 1 ];
						if ( not descripteur_.estUnePieceVide( nomPiece ) )
						{
							AnnuairePieces::iterator cible = pieces_.find( nomPiece );
							if ( cible == pieces_.end() )
							{
								cible = pieces_.insert( AnnuairePieces::value_type( nomPiece, new Piece( nomPiece, plateau_ ) ) ).first;
							}
							cible->second->completerGeometrie( position );
						}
						else
						{
							plateau_( position ) = 0;
						}
					}
				}
			}

			// On stocke l'âne rouge pour éviter de le rechercher par la suite.
			
			AnnuairePieces::iterator trouvaille = pieces_.find( descripteur_.nomAneRouge );
			if ( trouvaille != pieces_.end() )
			{
				pAneRouge_ = trouvaille->second;
			}
			else
			{
				throw runtime_error( "Flux de données malformé : pas d'âne rouge trouvé sur le plateau" );
			}
		}
		catch ( runtime_error & erreur )
		{
			nettoyer();
			throw erreur;
		}

		return entree;
	}
	
	ostream & Jeu::serialiser( std::ostream & sortie ) const
	{
		for ( size_t y = 1; y < plateau_.hauteur() - 1; ++y ) {
			for ( size_t x = 1; x < plateau_.largeur() - 1; ++x ) {
				Piece * pPiece = plateau_( Position( x, y ) );
				cout << (pPiece ? pPiece->getNom() : '.');
			}
			cout << endl;
		}
		return sortie;
	}
	
	void Jeu::nettoyer()
	{
		ITERER_CONST( AnnuairePieces, pieces_, itr )
		{
			// Les pièces effacées se désenregistrent elles-mêmes du plateau.
			delete itr->second;
		}
		pieces_.clear();
	}

	void Jeu::recupererPiecesMobiles( std::vector< PieceMobile * > & piecesMobiles ) const
	{
		ITERER_CONST( AnnuairePieces, pieces_, itr )
		{
			const char nom = itr->first;
			if ( not descripteur_.estUnePieceVide( nom ) and nom != descripteur_.nomBordure )
			{
				piecesMobiles.push_back( itr->second );
			}
		}
	}

	bool Jeu::estResolu() const
	{
		Position centreBas( plateau_.largeur() / 2, plateau_.hauteur() - 2 );
		if ( plateau_( centreBas ) != pAneRouge_ ) return false;
		if ( plateau_( centreBas.decaler( OUEST ) ) != pAneRouge_ ) return false;
		return true;
	}

	string Jeu::Plateau::calculerEmpreinte() const
	{
		/*
		On calcule l'empreinte qui est juste l'image simplifiée du plateau
		de jeu. Comme on ne mémorise que le genre des pièces, intervertir
		deux pièces de même genre ne change pas l'empreinte. Il est aussi
		facile de voir qu'un ensemble de cases de même genre ne peut être
		occupé que par un unique assemblage de pièces. Ce ne serait pas le
		cas si, par exemple, les barres horizontales et verticales avaient
		le même genre.
		*/
		string empreinte;
		empreinte.reserve( (hauteur_ - 2) * (largeur_ - 2) );
		for ( size_t y = 1; y < hauteur_ - 1; ++y )
		{
			for ( size_t x = 1; x < largeur_ - 1; ++x )
			{
				Piece * pPiece = (*this)( Position( x, y ) );
				if ( pPiece )
				{
					empreinte += pPiece->getGenre();
				}
				else
				{
					empreinte += '.';
				}
			}
		}
		
		/*
		Puis sa symétrie selon l'axe vertical.
		*/
		string etnierpme = empreinte;
		size_t saut = largeur_ - 2;
		for ( size_t y = 1; y < hauteur_ - 1; ++y )
		{
			reverse( &empreinte[ (y - 1) * saut ], &empreinte[ y * saut ] );
		}

		/*
		Et on retourne la plus "forte" des deux. De cette manière, les
		empreintes de deux plateaux symétriques seront toujours identiques.
		*/
		return empreinte > etnierpme ? empreinte : etnierpme;
	}
	
	Jeu::Piece::~Piece()
	{
		ITERER( TypeGeometrie, geometrie_, itrPosition )
		{
			plateau_( *itrPosition ) = 0;
		}
	}

	void Jeu::Piece::completerGeometrie( const Position & position )
	{
		geometrie_.push_back( position );
		plateau_( position ) = this;
		genre_ = '\0'; // Force le recalcule du genre.
	}

	bool Jeu::Piece::estUneBarreHorizontale() const
	{
		Position distance = geometrie_[0].distance( geometrie_[1] );
		return distance.x_ == 1 and distance.y_ == 0;
	}

	bool Jeu::Piece::estUneBarreVerticale() const
	{
		Position distance = geometrie_[0].distance( geometrie_[1] );
		return distance.x_ == 0 and distance.y_ == 1;
	}

	/*
	La distance entre n'importe quelle case d'un carré de 4 cases et son
	centre est de (0.5, 0.5). Comme nos coordonnées sont entières, on
	multiplie tout par 2 afin que cette distance soit de (1, 1).
	*/
	bool Jeu::Piece::estUnCarre() const
	{
		Position centre( 0, 0 );

		ITERER_CONST( TypeGeometrie, geometrie_, itr1 )
		{
			centre.x_ += itr1->x_ * 2;
			centre.y_ += itr1->y_ * 2;
		}
		centre.x_ /= geometrie_.size();
		centre.y_ /= geometrie_.size();

		ITERER_CONST( TypeGeometrie, geometrie_, itr2 )
		{
			Position distance = centre.distance( Position( itr2->x_ * 2, itr2->y_ * 2 ) );
			if ( distance.x_ != 1 or distance.y_ != 1 )
			{
				return false;
			}
		}
		return true;
	}

	char Jeu::Piece::getGenre() const
	{
		// On calcule le genre de manière paresseuse.
		if ( genre_ == '\0' )
		{
			size_t surface = geometrie_.size(); 
			if ( surface == 1 )
			{
				genre_ = 'P';
			}
			else if ( surface == 2 and estUneBarreVerticale() )
			{
				genre_ = 'V';
			}
			else if ( surface == 2 and estUneBarreHorizontale() )
			{
				genre_ = 'H';
			}
			else if ( surface == 4 and estUnCarre() )
			{
				genre_ = 'C';
			}
			else
			{
				throw logic_error( "Géométrie non reconnue" );
			}
		}
		return genre_;
	}

	bool Jeu::Piece::deplacer( DIRECTION direction )
	{
		Position nouvelleOrigine = origine_.decaler( direction );
		
		// On vérifie que les cases sur le chemin de la pièce sont libres.
		ITERER( TypeGeometrie, geometrie_, itrPosition )
		{
			Piece * pPiece = plateau_( itrPosition->decaler( nouvelleOrigine ) );
			if ( pPiece and pPiece != this )
			{
				// S'il n'y a pas la place, on ne déplace pas la pièce. 
				return false;
			}
		}

		// On enlève la pièce du plateau.
		ITERER( TypeGeometrie, geometrie_, itrPositionAvant )
		{
			plateau_( itrPositionAvant->decaler( origine_ ) ) = 0;
		}
		
		origine_ = nouvelleOrigine;

		// Pour la remettre à sa nouvelle position.
		ITERER( TypeGeometrie, geometrie_, itrPositionApres )
		{
			plateau_( itrPositionApres->decaler( origine_ ) ) = this;
		}
		
		return true;
	}

	std::istream & operator >> ( std::istream & entree, Jeu & jeu )
	{
		return jeu.deserialiser( entree );
	}
	
	std::ostream & operator << ( std::ostream & sortie, const Jeu & jeu )
	{
		return jeu.serialiser( sortie );
	}

} // namespace AneRouge
