// UTF-8 encoding
// Copyright 2009 Florent Lioult.

/**
 * @file	Jeu.hpp
 * @author	Florent Lioult
 * @date	2009
 */

#ifndef ANE_ROUGE__JEU_HPP
#define ANE_ROUGE__JEU_HPP

#include <string>
#include <vector>
#include <map>
#include <iostream>

#include "macro.hpp"
#include "DescripteurJeu.hpp"
#include "Position.hpp"

namespace AneRouge
{
	/**
	 * @brief	Une pièce mobile.
	 */
	class PieceMobile
	{
	public:
		virtual ~PieceMobile() {}
		
		virtual char	getNom() const = 0;
		
		/**
		 * @brief	Tente de déplacer d'une case cette pièce sur son plateau
		 *			de jeu.
		 * @param	direction La direction vers laquelle déplacer cette pièce.
		 * @return	Le déplacement a-t-il été possible ?
		 */
		virtual bool	deplacer( DIRECTION direction ) = 0;
	};

	/**
	 * @brief	Le jeu de l'âne rouge formée d'un plateau et de pièces dont
	 *			certaines sont mobiles.
	 */
	class Jeu
	{
		NON_COPIABLE( Jeu )

	public:
		Jeu( const DescripteurJeu & );
		virtual ~Jeu();
		
		/**
		 * Un jeu peut être sérialisé/désérialisé dans un format texte simple.
		 */
		// @{
		std::istream &	deserialiser( std::istream & entree ); 
		std::ostream &	serialiser( std::ostream & sortie ) const; 
		// @}
		
		/**
		 * @brief	Renseigne une collection avec la liste des pièces autorisées à
		 *			bouger.
		 *
		 * Pour être précis, cette méthode aurait du s'appeler
		 * recupererPiecesMobilesAyantLeDroitDeBouger... Sachant
		 * que toutes les pièces sont mobiles par ailleurs.
		 */
		void		recupererPiecesMobiles( std::vector< PieceMobile * > & ) const;
		
		/**
		 * @brief	Ce foutu âne a-t-il atteint la sortie ?
		 *
		 * Comme on gère des plateaux de taille quelconque et des ânes rouges de
		 * forme quelconque aussi, le critère de résolution est difficile à
		 * établir. En pratique on considèrera que les deux cases centrées au bas
		 * du plateau doivent être occupées par le bourricot.
		 *
		 * Petite parenthèse : si la bordure du plateau avait fait partie des
		 * données sérialisées, il aurait été possible d'indiquer la "porte" du
		 * puzzle pour l'âne.
		 */
		bool		estResolu() const;
		
		/**
		 * @brief	Calcule l'empreinte du jeu dans son état actuel.
		 *
		 * Deux jeux donnés sont dans des états équivalents ssi ils ont la même
		 * empreinte. Par équivalent on entend deux plateaux de jeu indissociables
		 * modulo la symétrie axiale verticale.
		 */
		std::string	calculerEmpreinte() const { return plateau_.calculerEmpreinte(); }
		
	private:
		class Piece;
		
		/**
		 * @brief	Le plateau de jeu.
		 *
		 * Un tableau 2D où chaque case pointe sur l'éventuelle pièce qui l'occupe.
		 * Un même pièce occupe généralement plusieurs cases simultanéement. Précisons
		 * aussi qu'il n'existe pas de pièces pour les emplacements vides. C'est assez
		 * logique mais on perd quand même l'information sur la localisation exacte
		 * des "non pièces" 'A' et 'B' (qui sont toutes deux remplacées par des '.'
		 * lors d'une sérialisation).
		 *
		 * Le plateau de jeu qu'on manipule a toujours une taille plus grande que
		 * celle définie dans les fichiers texte de manière à se ménager de la
		 * place pour une bordure. Cette bordure, occupée par une pièce immobile,
		 * permet de se passer des tests aux limites quand on vérifie si une pièce
		 * mobile peut se déplacer dans une direction donnée. Le but est davantage
		 * de simplifier le code que de l'optimiser.
		 */
		class Plateau
		{
			NON_COPIABLE( Plateau )

		public:
			Plateau() :
				largeur_( 0 ),
				hauteur_( 0 )
			{}

			Plateau( std::size_t largeur, std::size_t hauteur ) :
				largeur_( largeur ),
				hauteur_( hauteur ),
				contenu_( largeur * hauteur )
			{}

			void redimensionner( std::size_t largeur, std::size_t hauteur )
			{
				largeur_ = largeur;
				hauteur_ = hauteur;
				contenu_.clear();
				contenu_.resize( largeur * hauteur );
			}

			std::size_t largeur() const { return largeur_; }
			std::size_t hauteur() const { return hauteur_; }

			/**
			 * Un plateau est essentiellement un conteneur 2D a accès direct.
			 */
			// @{
			Piece *		operator() ( const Position & p ) const	{ return contenu_[ p.x_ + p.y_ * largeur_ ]; }
			Piece * &	operator() ( const Position & p )		{ return contenu_[ p.x_ + p.y_ * largeur_ ]; }
			// @}
			
			std::string calculerEmpreinte() const;
			
		private:
			std::size_t largeur_;
			std::size_t hauteur_;
			std::vector< Piece * > contenu_;
		};

		/**
		 * @brief	Une pièce du jeu.
		 *
		 * Une pièce est un ensemble de positions qui définissent sa géométrie et
		 * qu'on sait faire bouger en bloc.
		 */
		class Piece : public PieceMobile
		{
			NON_COPIABLE( Piece )
			
		public:
			/**
			 * Construit un pièce avec une géometrie vide.
			 */
			Piece( char nom, Plateau & plateau ) :
				nom_( nom ),
				genre_( '\0' ),
				plateau_( plateau ),
				origine_( 0, 0 )
			{}

			virtual ~Piece();
			
			/**
			 * Ajoute une case à la géometrie d'une pièce.
			 */
			void completerGeometrie( const Position & position );
			
			/**
			 * Le genre d'une pièce définit de quel type de pièce il s'agit en tenant
			 * compte de son orientation spatiale (une barre verticale et une barre
			 * horizontale n'ont pas le même genre par exemple).
			 */
			inline char getGenre() const;

		public: // PieceMobile
			inline char getNom() const { return nom_; };
			inline bool deplacer( DIRECTION direction );

		private:
			bool estUneBarreHorizontale() const;
			bool estUneBarreVerticale() const;
			bool estUnCarre() const;

		private:
			typedef std::vector< Position > TypeGeometrie;

			char nom_;
			mutable char genre_; // Mutable car calculée de manière paresseuse.
			Plateau & plateau_;
			TypeGeometrie geometrie_;

			/**
			 * Plutôt que de mettre à jour toutes les positions de la géometrie,
			 * on se contente de modifier son origine (initialement fixée à 0x0).
			 */
			Position origine_;
		};

	private:
		/**
		 * Vide le plateau de ses pièces.
		 */
		void nettoyer();
	
	private:
		/// Nom d'une pièce -> instance correspondante.
		typedef std::map< char, Piece * > AnnuairePieces;

		DescripteurJeu descripteur_;
		Plateau plateau_;
		AnnuairePieces pieces_;
		Piece * pAneRouge_;
	};

	std::istream & operator >> ( std::istream & entree, Jeu & jeu );
	std::ostream & operator << ( std::ostream & sortie, const Jeu & jeu );
}

#endif // ANE_ROUGE__JEU_HPP
