// UTF-8 encoding
// Copyright 2009 Florent Lioult.

/**
 * @file	Partie.cpp
 * @author	Florent Lioult
 * @date	2009
 */

#include <ctime>
#include <stack>
#include <fstream>

#include "macro.hpp"
#include "Partie.hpp"

using namespace std;

namespace AneRouge {

	void Partie::chargerJeu( const char * cheminFichier )
	{
		/*
		On initialise le jeu avec un fichier décrivant l'agencement des pièces.
		*/
		ifstream fichier( cheminFichier );
		if ( fichier.is_open() )
		{
			try
			{
				fichier >> jeu_;
				fichier.close();
				cout << jeu_ << endl;
			}
			catch ( const runtime_error & erreur )
			{
				fichier.close();
				throw erreur;
			}
		}
		else
		{
			throw runtime_error( "Fichier introuvable" );
		}
		
		jeu_.recupererPiecesMobiles( piecesMobiles_ );
		
		/*
		On génère une bonne fois pour toute l'ensemble des coups à tester pour
		chaque configuration de jeu.
		*/
		for ( size_t i = 0; i < piecesMobiles_.size(); ++i )
		{
			for ( int d = 0; d < CARD_DIRECTION; ++d )
			{
				tousLesCoups_.push_back( Coup( i, static_cast< DIRECTION >( d ) ) );
			}
		}
	}

	Partie::Configuration * Partie::getConfiguration( const string & empreinte )
	{
		bool estNouvelle;
		return getConfiguration( empreinte, estNouvelle );
	}
	
	Partie::Configuration * Partie::getConfiguration( const string & empreinte, bool & estNouvelle )
	{
		map< string, Configuration * >::iterator trouvaille = univers_.find( empreinte );
		if ( trouvaille != univers_.end() )
		{
			estNouvelle = false;
			return trouvaille->second;
		}
		else
		{
			estNouvelle = true;
			Configuration * pConfiguration = new Configuration( empreinte );
			univers_.insert( map< string, Configuration * >::value_type( empreinte, pConfiguration ) );
			return pConfiguration;
		}
	}

	void Partie::jouer()
	{
		clock_t debut = clock();
		{
			vector< Configuration * > solutions;
			
			/*
			La résolution du jeu se fait essentiellement par la force brute en 3 étapes :
			
			1)	Construction du graphe des configurations de jeu atteignables les unes
				à partir des autres en partant de la configuration initiale (du coup on
				peut ne pas énumérer toutes les configurations existantes, juste celles
				atteignables).
			*/
			Configuration * pConfigurationInitiale = getConfiguration( jeu_.calculerEmpreinte() );
			explorer( pConfigurationInitiale, solutions );
			cout << univers_.size() << " configuration(s) distincte(s) trouvée(s)." << endl;
			
			/*
			La seule optimisation de cet algorithme réside dans la définition d'une configuration
			qui agrége tous les plateaux de jeu qui ne sont pas distinguables abstraction faite du
			nom des pièces et de la symétrie axiale verticale.
			
			De fait, le graphe de configurations construit possède des transitions qui agrégent
			plusieurs coups sachant qu'une configuration désigne plusieurs plateaux réels et qu'un
			plateau peut mener à une même configuration par différents coups. C'est d'ailleurs la
			raison pour laquelle la restitution de la solution une fois le plus court chemin trouvé
			nécessite de retrouver une suite de coups réels.
			*/
			
			if ( not solutions.empty() )
			{
				/*
				2)	Marquage du graphe avec la distance la plus courte vers une solution
					(la première étape permettant d'en faire l'inventaire).
				*/
				ITERER( vector< Configuration * >, solutions, itr ) marquer( *itr );
				
				cout
					<< "La meilleur solution comporte "
					<< pConfigurationInitiale->distance_ << " coup(s)." << endl;
				/*
				3)	Traversée rapide du graphe en suivant le marquage menant au plus vite
					à une solution et restitution dans la foulée des coups à jouer.
				*/
				traverser( pConfigurationInitiale );
			}
			else
			{
				/*
				2b)	On est brocouille...
				*/
				cout << "Aucune solution trouvée." << endl;
			}
		}
		clock_t fin = clock();
		clock_t duree = fin - debut;
		
		cout << "Durée : " << duree * 1.0 / CLOCKS_PER_SEC << " seconde(s)." << endl;
		
		ITERER( TypeUnivers, univers_, itr ) delete itr->second;
	}

	/**
	 * On explore en profondeur d'abord l'espace des configurations possibles du jeu.
	 * Contrairement à la méthode de marquage, la récursivité n'est pas coûteuse car
	 * on ne visite une configuration donnée qu'une seule et unique fois (ce qui n'est
	 * pas le cas lors de la mise à jour de la plus courte distance).
	 *
	 * Une autre raison/caractéristique de cette approche est qu'on ne mémorise pas les
	 * différents plateaux de jeu rencontrés mais juste la pile de coups menant au
	 * plateau courant. La version en largeur d'abord serait plus délicate à élaborer.
	 */
	void Partie::explorer( Configuration * pConfiguration, vector< Configuration * > & solutions )
	{
		if ( jeu_.estResolu() )
		{
			solutions.push_back( pConfiguration );
		}

		ITERER_CONST( vector< Coup >, tousLesCoups_, itrCoup )
		{
			const Coup & coup = *itrCoup;
			
			if ( jouer( coup ) )
			{
				bool estNouvelle;
				Configuration * pConfigurationVoisine = getConfiguration( jeu_.calculerEmpreinte(), estNouvelle );
				
				pConfiguration->voisinage_.push_back( pConfigurationVoisine );
				
				if ( estNouvelle )
				{
					explorer( pConfigurationVoisine, solutions );
				}

				annuler( coup );
			}
		}
	}
	
	/**
	 * On innonde le graphe en mettant récusrivement à jour pour chaque configuration
	 * sa plus courte distance à une solution.
	 *
	 * Méthode rendue obsolète par la version ci-dessous.
	 */
	void Partie::marquer( Configuration * pConfiguration, size_t distance )
	{
		if ( pConfiguration->distance_ > distance )
		{
			pConfiguration->distance_ = distance;
			ITERER( vector< Configuration * >, pConfiguration->voisinage_, itr )
			{
				marquer( *itr, distance + 1 );
			}
		}
	}
	
	/**
	 * Même chose que ci-dessus mais en supprimant la coûteuse récursivité en profondeur
	 * d'abord, pas vraiment adaptée à une innondation de graphe. Grosso modo cette
	 * version non récursive et en largeur d'abord est 10 fois plus rapide que la
	 * version récursive mais sans doute 4 fois moins lisible...
	 */
	void Partie::marquer( Configuration * pConfigurationDepart )
	{
		stack< Configuration * > configurations1;
		stack< Configuration * > configurations2;
		stack< Configuration * > * pPile1 = &configurations1;
		stack< Configuration * > * pPile2 = &configurations2;
		
		pPile1->push( pConfigurationDepart );
		size_t distance = 0;
		
		/*
		La pile N contient toutes les configurations à la distance D dont on récupére les
		voisins directs afin de contruire la pile N+1 contenant ainsi les configurations
		à la distance D+1.
		
		D'une itération sur l'autre on recycle la première pile (qu'on a vidée en l'explorant)
		en l'intervertissant simplement avec la seconde pile.
		*/

		while ( not pPile1->empty() )
		{
			while ( not pPile1->empty() )
			{
				Configuration * pConfiguration = pPile1->top();
				pPile1->pop();
				
				if ( pConfiguration->distance_ > distance )
				{
					pConfiguration->distance_ = distance;
					ITERER( vector< Configuration * >, pConfiguration->voisinage_, itr )
					{
						pPile2->push( *itr );
					}
				}
			}
			++distance;
			swap( pPile1, pPile2 );
		}
	}
	
	/**
	 * On traverse le graphe de configurations dûment marquées avec leur distance la plus courte
	 * à une solution.
	 */
	void Partie::traverser( Configuration * pConfiguration )
	{
		ITERER_CONST( vector< Configuration * >, pConfiguration->voisinage_, itrVoisin )
		{
			if ( (*itrVoisin)->distance_ < pConfiguration->distance_ )
			{
				/*
				La seule complexité du parcour est ici. Comme les empreintes désignent non pas
				un unique plateau de jeu mais plusieurs plateaux équivalents, il n'existe pas
				de transition unique sous forme d'un coup précis à jouer. On est donc obligé
				à chaque configuration atteinte de retester tous les coups possibles jusqu'à
				en trouver un pour atteindre la prochaine empreinte. Il peut d'ailleurs en
				exister plusieurs mais ils seront tous équivalents en distance parcourue.
	 			*/
				const string & empreinteRecherchee = (*itrVoisin)->empreinte_;
				
				ITERER_CONST( vector< Coup >, tousLesCoups_, itrCoup )
				{
					const Coup & coup = *itrCoup;
					
					if ( jouer( coup ) )
					{
						if ( jeu_.calculerEmpreinte() == empreinteRecherchee )
						{
							cout
								<< "Déplacer la pièce " << piecesMobiles_[ coup.indexMobile_ ]->getNom()
								<< " vers " << LIBELLES_DIRECTIONS[ coup.direction_ ] << '.' << endl;
							traverser( *itrVoisin  );
							return;
						}
						annuler( coup );
					}
				}
				
				throw logic_error( "Impossible de trouver un coup pour passer d'une configuration à sa voisine." );
			}
		}
		
		if ( not jeu_.estResolu() )
		{
			throw logic_error( "La traversée n'a pas menée à une solution" );
		}
	}

} // namespace AneRouge
