// UTF-8 encoding
// Copyright 2009 Florent Lioult.

/**
 * @file	Partie.hpp
 * @author	Florent Lioult
 * @date	2009
 */

#ifndef ANE_ROUGE__PARTIE_HPP
#define ANE_ROUGE__PARTIE_HPP

#include <string>
#include <vector>
#include <map>

#include "macro.hpp"
#include "Jeu.hpp"

namespace AneRouge
{
	/**
	 * @brief	Une partie (réussie en principe) du jeu de l'âne rouge.
	 */
	class Partie
	{
		NON_COPIABLE( Partie )

	public:
		/**
		 * @brief	Joue une partie de l'âne rouge en tentant de trouver
		 *			la plus meilleur solution, c'est à dire celle comportant
		 *			le moins de coups.
		 * @param	cheminFichier Le chemin du fichier texte définissant
		 *			le plateau de jeu.
		 */
		static void jouer( const char * cheminFichier )
		{
			Partie partie;
			partie.chargerJeu( cheminFichier );
			partie.jouer();
		}
		
	private:
		/**
		 * @brief	Un coup identifie une pièce à jouer et la direction dans
		 *			laquelle la déplacer.
		 */
		struct Coup
		{
			std::size_t indexMobile_;
			DIRECTION direction_;
			
			Coup() :
				indexMobile_( std::numeric_limits< size_t >::max() ),
				direction_( NORD ) {}
			
			Coup( std::size_t indexMobile, DIRECTION direction ) :
				indexMobile_( indexMobile ),
				direction_( direction ) {}
		};

		/**
		 * @brief	Une configuration de jeu donné unifiant tous les plateaux
		 *			de même empreinte.
		 */
		struct Configuration
		{
			/// L'empreinte de cette configuration de plateaux.
			std::string empreinte_;
			
			/// La distance de cette configuration à la plus proche solution.
			std::size_t distance_;
			
			/// Les configurations qu'il est possible d'atteindre en un seul coup à partir de celle-ci.
			std::vector< Configuration * > voisinage_;
			
			Configuration( const std::string & empreinte ) :
				empreinte_( empreinte),
				distance_( std::numeric_limits< size_t >::max() )
			{}
		};
		
	private:
		Partie() : jeu_( DescripteurJeu::creerDescripteurParDefaut() ) {}

		void jouer();

		bool jouer		( const Coup & coup ) { return piecesMobiles_[ coup.indexMobile_ ]->deplacer( coup.direction_ ); }
		bool annuler	( const Coup & coup ) { return piecesMobiles_[ coup.indexMobile_ ]->deplacer( DEMI_TOURS[ coup.direction_ ] ); }

		void chargerJeu( const char * cheminFichier );
		
		/**
		 * @brief	Retourne la configuration correspondant à une empreinte
		 *			en la créant au besoin.
		 */
		Configuration * getConfiguration( const std::string & );

		/**
		 * @brief	Retourne la configuration correspondant à une empreinte
		 *			en la créant au besoin.
		 * @param	estNouvelle Indiquera en sortie de méthode si la configuration
		 *			retournée a été nouvellement créée.
		 */
		Configuration * getConfiguration( const std::string &, bool & estNouvelle );
		
		void	explorer	( Configuration *, std::vector< Configuration * > & );
		void	marquer		( Configuration * pConfiguration, size_t distance );
		void	marquer		( Configuration * pConfiguration );
		void	traverser	( Configuration * pConfiguration );

	private:
		typedef std::map< std::string, Configuration * > TypeUnivers;

		Jeu jeu_;

		/// les pièces qu'il est possible de déplacer dans le jeu.
		std::vector< PieceMobile * > piecesMobiles_;
		
		/// La liste précalculée de tous le coups possibles (nombre de pièces x nombre de directions).
		std::vector< Coup > tousLesCoups_;
		
		/// L'espace de jeu contenant toutes les configurations explorées.
		TypeUnivers univers_;
	};
}

#endif // ANE_ROUGE__PARTIE_HPP
